// WebSocket 全局管理器
import webAPI from '@/api/webAPI'

class WebSocketManager {
  constructor() {
    this.websocket = null
    this.wsConnected = false
    this.wsMessages = []
    this.ukeyStatus = 'unknown'
    this.ukeyLastUpdate = null
 
    this.operatorOptions = []
    this.callbacks = {
      onConnect: [],
      onDisconnect: [],
      onUKeyStatus: [],
      onHwCode: [],
      onMessage: []
    }
  }

  // 初始化 WebSocket 连接
  initWebSocket() {
    if (this.websocket && this.wsConnected) {
      console.log('WebSocket 已连接，无需重复连接')
      return
    }

    try {
      this.websocket = new WebSocket('ws://127.0.0.1:8889')
      
      // 连接成功
      this.websocket.onopen = (event) => {
        console.log('WebSocket 连接已建立:', event)
        this.wsConnected = true
        
        // 发送获取硬件代码的请求
        // this.sendMessage({
        //   action: 'get_hwcode'
        // })
        
        // 触发连接回调
        this.triggerCallbacks('onConnect', { event })
      }
      
      // 接收消息
      this.websocket.onmessage = (event) => {
        console.log('收到 WebSocket 消息:', event.data)
        try {
          const data = JSON.parse(event.data)
          this.wsMessages.push({
            ...data,
            receivedAt: new Date().toISOString()
          })
          
          // 处理不同类型的消息
          this.handleMessage(data)
          
          // 触发消息回调
          this.triggerCallbacks('onMessage', { data, event })
        } catch (error) {
          console.error('解析 WebSocket 消息失败:', error)
          this.wsMessages.push({
            raw: event.data,
            receivedAt: new Date().toISOString()
          })
        }
      }
      
      // 连接错误
      this.websocket.onerror = (error) => {
        console.error('WebSocket 连接错误:', error)
        this.wsConnected = false
        this.triggerCallbacks('onDisconnect', { error })
      }
      
      // 连接关闭
      this.websocket.onclose = (event) => {
        console.log('WebSocket 连接已关闭:', event)
        this.wsConnected = false
        this.triggerCallbacks('onDisconnect', { event })
        
        // 尝试重连
        setTimeout(() => {
          if (!this.wsConnected) {
            console.log('尝试重新连接 WebSocket...')
            this.initWebSocket()
          }
        }, 3000)
      }
      
    } catch (error) {
      console.error('创建 WebSocket 连接失败:', error)
    }
  }

  // 处理消息
  handleMessage(data) {
    // 根据 type 字段判断是否是 ukey 状态变化消息
    if (data.type === 'ukey_popup.remove' || data.type === 'ukey_popup.insert') {
      this.handleUKeyStatus(data)
    }
  }

  // 处理 UKey 状态变化
  handleUKeyStatus(data) {
    console.log('UKey 状态变化:', data)
    
    // 根据 type 字段判断状态
    let status = ''
    if (data.type === 'ukey_popup.remove') {
      status = 'removed'
    } else if (data.type === 'ukey_popup.insert') {
      status = 'inserted'
    } else {
      console.log('未知的 UKey 状态类型:', data.type)
      return
    }
    
    // 检查状态是否真的发生了变化
    // if (this.ukeyStatus === status) {
    //   console.log('UKey 状态未变化，跳过提示')
    //   return
    // }
    
    // 更新 UKey 状态
    this.ukeyStatus = status
    this.ukeyLastUpdate = new Date()
    
    const payload = data && data.data ? data.data : {}

    // 根据 UKey 状态执行相应操作
    if (status === 'removed') {
      // UKey 拔出时清除数据
      this.handleUKeyRemoved(payload)
    } else if (status === 'inserted') {
      // UKey 插入时重新请求硬件代码
      this.handleUKeyInserted(payload)
    }
    
    // 触发 UKey 状态回调，传递格式化后的数据
    this.triggerCallbacks('onUKeyStatus', { 
      data: {
        ...data.data,
        status: status,
        type: data.type
      },
      originalData: data
    })
  }

  // 处理 UKey 拔出
  handleUKeyRemoved(payload = {}) {
    console.log('UKey 已拔出，准备处理状态', payload)

    const currentUkeySn = (sessionStorage.getItem('ukeySn') || '').trim()
    const removedUkeySn = ((payload.ukeySn || payload.sn || payload.deviceSn || '') + '').trim()

    if (!currentUkeySn) {
      console.warn('当前会话未记录登录使用的 UKey，跳过自动退出')
      return
    }

    if (!removedUkeySn) {
      console.warn('WebSocket 消息未包含 UKey 序列号，跳过自动退出')
      return
    }

    if (currentUkeySn !== removedUkeySn) {
      console.log('拔出的 UKey 非当前登录使用的 UKey，保持会话', {
        currentUkeySn,
        removedUkeySn
      })
      return
    }

    console.log('检测到当前登录使用的 UKey 被拔出，即将调用退出登录接口')

    // 调用退出登录接口
    webAPI.webAuthLogout().then(res => {
      console.log('退出登录接口返回结果:', res)
      // 无论接口是否成功，都清除本地数据并跳转
      this.clearSessionAndNavigate()
    }).catch(error => {
      console.error('退出登录接口调用失败:', error)
      // 即使接口调用失败，也清除本地存储并跳转
      this.clearSessionAndNavigate()
    })
  }

  // 处理 UKey 插入
  handleUKeyInserted(payload = {}) {
    console.log('UKey 已插入，重新请求硬件代码')
    
    // 如果 WebSocket 已连接，发送获取硬件代码请求
    if (this.wsConnected && this.websocket) {
      // console.log('发送获取硬件代码请求')
      // this.sendMessage({ action: 'get_hwcode' })
    } else {
      // console.log('WebSocket 未连接，无法发送硬件代码请求')
    }
  }

  // 处理硬件代码数据
  handleHwCodeData(data) {
    console.log('收到硬件代码数据:', data)
    
    if (data.data) {
      // console.log('ukey数据:', data.data)
      this.operatorOptions = [{
        label: data.data,
        value: data.data
      }]
      
      // 触发硬件代码回调
      this.triggerCallbacks('onHwCode', { data })
    }
  }

  // 处理硬件代码响应
  handleHwCodeResponse(data) {
    console.log('收到硬件代码响应:', data)
    
  }

  // 发送消息
  sendMessage(message) {
    if (this.websocket && this.wsConnected) {
      try {
        this.websocket.send(JSON.stringify(message))
        console.log('发送 WebSocket 消息:', message)
      } catch (error) {
        console.error('发送 WebSocket 消息失败:', error)
      }
    } else {
      console.warn('WebSocket 未连接，无法发送消息')
    }
  }

  // 关闭连接
  closeWebSocket() {
    if (this.websocket) {
      this.websocket.close()
      this.websocket = null
      this.wsConnected = false
      console.log('WebSocket 连接已关闭')
    }
  }

  // 获取当前路由
  getCurrentRoute() {
    // 这里需要根据你的路由系统来获取当前路由
    // 如果是 Vue Router，可以通过 this.$router.currentRoute 获取
    return window.location ? { path: window.location.pathname } : null
  }

  // 跳转到登录页
  navigateToLogin() {
    // 这里需要根据你的路由系统来跳转
    // 如果是 Vue Router，可以通过 this.$router.push('/login') 跳转
    if (window.location) {
      window.location.href = '/sys/login'
    }
  }

  // 设置路由跳转函数（由应用初始化时调用）
  setRouter(router) {
    this.router = router
  }

  // 使用 Vue Router 跳转
  navigateToLoginWithRouter() {
    if (this.router) {
      this.router.push('/sys/login')
    } else {
      this.navigateToLogin()
    }
  }

  // 清除会话数据并跳转到登录页
  clearSessionAndNavigate() {
    // 清除硬件代码相关数据
    this.operatorOptions = []

    // 清除会话中的登录信息
    sessionStorage.removeItem('username')
    sessionStorage.removeItem('ukeySn')
    sessionStorage.removeItem('token')
    sessionStorage.removeItem('clientRandom')
    sessionStorage.removeItem('clientSign')

    // 获取当前路由
    const currentRoute = this.getCurrentRoute()

    // 如果不在登录页，则跳转到登录页
    if (currentRoute && currentRoute.path !== '/sys/login') {
      console.log('跳转到登录页')
      this.navigateToLoginWithRouter()
    }
  }

  // 注册回调
  on(event, callback) {
    if (this.callbacks[event]) {
      this.callbacks[event].push(callback)
    }
  }

  // 移除回调
  off(event, callback) {
    if (this.callbacks[event]) {
      const index = this.callbacks[event].indexOf(callback)
      if (index > -1) {
        this.callbacks[event].splice(index, 1)
      }
    }
  }

  // 触发回调
  triggerCallbacks(event, data) {
    if (this.callbacks[event]) {
      this.callbacks[event].forEach(callback => {
        try {
          callback(data)
        } catch (error) {
          console.error(`回调执行失败 (${event}):`, error)
        }
      })
    }
  }

  // 获取状态
  getStatus() {
    return {
      wsConnected: this.wsConnected,
      ukeyStatus: this.ukeyStatus,
      ukeyLastUpdate: this.ukeyLastUpdate,
    
      operatorOptions: this.operatorOptions
    }
  }
}

// 创建全局实例
const wsManager = new WebSocketManager()

export default wsManager
